// ModuleInterface.cpp: implementation of the CModuleInterface class.
// Version 3.1    6 Setember 2007 Rewrite to use serial port
// Version 3.2   16 November 2007 File made common to all USB modules
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ModuleInterface.h"
#include "serialport.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CModuleInterface::CModuleInterface()
{
	mDataPortBDirControl = 0;
	mDataPortCDirControl = 0;
	mDataPortDDirControl = 0;
}

CModuleInterface::CModuleInterface(CSerialPort *ModulePort)
{
	mDataPortBDirControl = 0;
	mDataPortCDirControl = 0;
	mDataPortDDirControl = 0;
	mPort = ModulePort;
}

CModuleInterface::~CModuleInterface()
{

}
///////////////////////////////////////////////////////////////////////////////
// Comm Port Access functions
///////////////////////////////////////////////////////////////////////////////

void CModuleInterface::CreateCommPort (int PortNum)
{
	//int   PortNum = Port;
	DWORD LinkSpeed = 9600;
	BYTE  NumDataBits = 8;

    mPort->Open(PortNum,
		 LinkSpeed,
		 CSerialPort::NoParity,
		 NumDataBits,
		 CSerialPort::OneStopBit,
		 CSerialPort::NoFlowControl);

    HANDLE hPort = mPort->Detach();
    mPort->Attach(hPort);
 }

///////////////////////////////////////////////////////////////////////////////
int CModuleInterface::ReadCommPort (void)
{
	DWORD dwRead = 0;
	char sRxBuf[10] = {0};

	mPort->Set0ReadTimeout();
	dwRead = mPort->Read(sRxBuf, 1);

    mPort->ClearReadBuffer();

    mPort->Flush();

    return sRxBuf[0];
}

///////////////////////////////////////////////////////////////////////////////
int CModuleInterface::WriteToCommPort (char * Buffer, int Length)
{
    mPort->Write(Buffer, (DWORD) Length);

    mPort->ClearWriteBuffer();
	return 0;
}

///////////////////////////////////////////////////////////////////////////////
void CModuleInterface::SendCommand (char ControlChar, int Value)
{
	// As only two charcters are sent a small buffer is enough
	char sBuff[3] = {0};
	
	sBuff[0] = ControlChar;
	sBuff[1] = Value;
	WriteToCommPort(sBuff,2);

}
/* End of Comm functions */


///////////////////////////////////////////////////////////////////////////////
// Generates the serial commands to open or close a relay.
// To ensure that previous opened or closed relays are not 
// changed the bit states are held as words in 2 arrays.
// Each element of the arrays corresponds to one of the
// control latches. The function uses the iBank value to
// select the appropriate latch and the uPosition to select
// which bit.
// The SendCommand is then called to pass the data to the serial port.
///////////////////////////////////////////////////////////////////////////////
bool CModuleInterface::ActivateRelay(int uPosition, char iBank, int OnOff )
{

    char uRData;
    if ( uPosition <= 8 )
    {
        // Position is under 8 so select word for lower relay bank
        if ( OnOff == 1 )
        {
            // closing relay so change correct bit to 1
			mcLowerRelay[iBank] = mcLowerRelay[iBank] | (1 << (uPosition - 1));
        }
        else
        {
            // opening relay so change correct bit to 0
            mcLowerRelay[iBank] = mcLowerRelay[iBank] & (~ (1 << (uPosition - 1)));
        }
    }    
    else
    {
         // Position is over 8 so subtract 8 and select word for lower relay bank
		uPosition = uPosition - 8;
        if ( OnOff == 1 )
        {
            // closing relay so change correct bit to 1
            mcUpperRelay[iBank] = mcUpperRelay[iBank] | (1 << (uPosition - 1));
        }
        else
        {
           // opening relay so change correct bit to 0
           mcUpperRelay[iBank] = mcUpperRelay[iBank] & (~(1 << (uPosition - 1)));
        }
    }
        
    
	// Change to char
	uRData = mcLowerRelay[iBank];

	SendCommand ('F', uRData);
    Sleep (10);
    uRData = mcUpperRelay[iBank];

    SendCommand('J', uRData);
    return true; 
}

///////////////////////////////////////////////////////////////////////////////
// This function constructs the commands to latch data
// across one of the relay control latches.
// This function relies upon initialisation to set all
// latch outputs low
void CModuleInterface::ActivateBank(char uBank)
{
   char BankWord;
   char uRData;
   
    //Prepare BankWord
    BankWord = 1 << (uBank);
    
    uRData = BankWord | 1;  // bank pos + OE High

    SendCommand ('C', uRData);
    
    //Return bank clock low for next access    
    uRData = 1;  //Clock low OE still high
    SendCommand ('C', uRData);
}


// Used to open all relays 
void CModuleInterface::ClearRelays(int NumRelays)
{
    char uRData;
    
    // Set Latch enable off
    uRData = 0;
    
    // Clear Relay control words
    for( int i = 1; i <= 3; i++)
    {
        mcLowerRelay[i] = 0;
        mcUpperRelay[i] = 0;
    }
    
    //set All relay lines low
	SendCommand ('C', uRData);
    Sleep (10);
    SendCommand ('F', uRData);  //All lower port Relays Low
    SendCommand ('J', uRData);  //All upper port Relays Low
    
	// This for the multiplexed version of modules
	if (NumRelays > 24)
    {
		//Set all latch ouputs low
		for (i = 1; i<= 6; i++)
		{
			uRData = (1 << i);
			SendCommand ('C', uRData);
    
			//Return bank clock low for next access
			uRData = 0;
			SendCommand ('C', uRData);
		}
		// Set Latch enable back to on
		uRData = 1;
		SendCommand ('C', uRData);
	}
}

///////////////////////////////////////////////////////////////////////////////
// Closes relay RelNum on bank RelBank
// The last parameter of ActivateRelay set to 1
// denotes close
void CModuleInterface::CloseRelay(int RelNum, int RelBank)
{
    ActivateRelay (RelNum, RelBank, 1);
    ActivateBank (RelBank);
    
}

///////////////////////////////////////////////////////////////////////////////
// Closes relay RelNum on bank RelBank
// The last parameter of ActivateRelay set to 1
// denotes close
void CModuleInterface::CloseRelay(int RelayPosition)
{
	int RelBank;
	int RelNum;

	RelBank = (RelayPosition / (int)(8));
	if((RelNum = RelayPosition % 8) == 0)
    {
		RelNum = 8;
	    RelBank = (RelayPosition / (int)(8));
	}
	else
	{
		RelBank = (RelayPosition / (int)(8)) + 1;
	}

	// Calculate Control character
	switch (RelBank)
	{
		case 1: 
			mcLowerRelay[1] = mcLowerRelay[1] | (1 << (RelNum - 1));
	        SendCommand ('C', mcLowerRelay[1]);
			break;
		case 2:
			mcLowerRelay[2] = mcLowerRelay[2] | (1 << (RelNum - 1));
	        SendCommand ('E', mcLowerRelay[2]);
			break;
		case 3:;
			mcLowerRelay[3] = mcLowerRelay[3] | (1 << (RelNum - 1));
	        SendCommand ('J', mcLowerRelay[3]);
			break;
		default: break;
    }
}

///////////////////////////////////////////////////////////////////////////////
// Open relay RelNum on bank RelBank
// The last parameter of ActivateRelay set to 0
// denotes open
void CModuleInterface::OpenRelay(int RelNum, int RelBank)
{
    ActivateRelay( RelNum, RelBank, 0);
    ActivateBank (RelBank);
           
}

///////////////////////////////////////////////////////////////////////////////
// Open relay RelNum on bank RelBank
// The last parameter of ActivateRelay set to 0
// denotes open
void CModuleInterface::OpenRelay(int RelayPosition)
{
	int RelBank;
	int RelNum;

	RelBank = (RelayPosition / (int)(8));
	if((RelNum = RelayPosition % 8) == 0)
    {
		RelNum = 8;
	    RelBank = (RelayPosition / (int)(8));
	}
	else
	{
		RelBank = (RelayPosition / (int)(8)) + 1;
	}


	switch (RelBank)
	{
		case 1: 
			mcLowerRelay[1] = mcLowerRelay[1] & (~(1 << (RelNum - 1)));
	        SendCommand ('C', mcLowerRelay[1]);
			break;
		case 2:
			mcLowerRelay[2] = mcLowerRelay[2] & (~(1 << (RelNum - 1)));
	        SendCommand ('E', mcLowerRelay[2]);
			break;
		case 3:;
			mcLowerRelay[3] = mcLowerRelay[3] & (~(1 << (RelNum - 1)));
	        SendCommand ('J', mcLowerRelay[3]);
			break;
		default: break;
    }
}

///////////////////////////////////////////////////////////////////////////////
int CModuleInterface::ReadDataPortVal (char Port)
{
    int PortValue;
        
    if (Port == 'B')
	{
		SendCommand ('A',0);
	}
    else
	{
        if (Port == 'C')
		{
			SendCommand ('D',0xF);
		}
	}   

    PortValue = ReadCommPort();

	return PortValue;
}

///////////////////////////////////////////////////////////////////////////////
// Sets the PIC ports to outputs or inputs
// Mask = bit high to change bit, low to not change bit
void CModuleInterface::SetPortStates(char Port, PortDirection Dir, unsigned char Mask)
{
   unsigned short DirMask;

   if ( Dir ==  InputState)
   {
      DirMask = 0xFF;
   }
   else
   {
       DirMask = 0;
   }
    /* The variable mDataPortXDirControl is modified each time this function is called
	 * The value of Mask defines which pins are to be changed and DirMask is set by
	 * the value of port direction.*/
	switch(Port)
	{
		case 'B':
			mDataPortBDirControl = ((~Mask) & mDataPortBDirControl) | (Mask & DirMask);
	        SendCommand ('B', mDataPortBDirControl); //Set B port
			break;
		case 'C':
			mDataPortCDirControl &= ((~Mask) & mDataPortCDirControl) | (Mask & DirMask);
	        SendCommand ('E', mDataPortCDirControl); //Set C port
			break;
		case 'D':
			mDataPortDDirControl &= ((~Mask) & mDataPortDDirControl) | (Mask & DirMask);
	        SendCommand ('H', mDataPortDDirControl); //Set D port
			break;
	}
	Sleep(10);
}

///////////////////////////////////////////////////////////////////////////////
// Called by other external classes to change
// the communcations port as Comm class is 
// only in scope to this class 
BOOL CModuleInterface::SetComPortNum(int ComPortNum)
{
	mPortNum = ComPortNum;
	return mPort->IsPortValid(ComPortNum);
}

/* End of file */
